import processing.sound.*;

/**
 * AudioCapture captures streaming audio from live input or from a file and processes it with PixelAudio.
 * 
 *   1. Launch the sketch. A display window appears with rainbow colors marking the signal path.
 *      The information at the top of the display indicates that "listening is false, source is file,
 *      the source is paused at 0/25000, and muting is false." 
 *      
 *   2. Press 'p' to turn the audio on and off. 
 *   
 *   3. While audio is playing, press the spacebar to start "listening", i.e., streaming audio 
 *      to audioBuffer and mapImage.
 *   
 *   4. Press 'm' to mute and unmute the audio. 
 *   
 *   5. While audio is muted, click in the display window. This triggers audio events. 
 *      You can trigger events while the audio is streaming, or just trigger them from 
 *      the captured buffer when audio is not streaming. You can "play" a muted stream 
 *      from a file by clicking at the right interval in one place. 
 *   
 *   6. Press 't' to change the audio source to "device", which by default is the built-in
 *      mic for your computer. If listening is off, you can still see the audio signal in
 *      the display. Press the spacebar to start capturing the audio source. 
 *      
 *   7. While the audio is streaming in from the computer mic, you can trigger audio events
 *      by clicking in the display window. The mic will pick up audio from your speakers, 
 *      a situation which can produce feedback. 
 *      
 * 
 * Using the Sound library in Processing, the control panel for Sound in MacOS, and the
 * BlackHole audio routing tool (https://existential.audio/blackhole/), you can obtain
 * a wide range of inputs and outputs, depending on your system software and hardware. 
 * YMMV, using Processing's Sound library. I'm using Minim. Some suggestions follow. 
 * 
 * For example, in MacOS: 
 *   Ignore Sound.inputDevice() and Sound.outputDevice(), use the System Settings instead.
 *   Set Output to BlackHole 16ch
 *   Set Input to your external audio hardware, for an external mic: mine is Volt2
 *   
 * Then in Max Audio Status control panel:
 *   Set Input Device to BlackHole 16ch
 *   Set Output Device to your external audio hardware, e.g. Volt2
 *   Create a patcher that gets signals from an adc~, route the adc~ to some sort of effects 
 *   (I just used a tapin~/tapout~ pair), pass the outputs to a gain~ and then to a dac~. 
 * 
 * I've provided a sample Max patcher, simpleAudioIO.maxpat, in the "data" folder for this sketch.
 * 
 * You can also configure this sketch to capture audio signals generated by another application. 
 * 
 * For example, using Max, open a patcher that generates and outputs audio, such as 
 * 05yFMSynthesis.maxpat in the MSP tutorial. 
 *   In MacOS Audio MIDI Setup tool, configure a Multi-Output Device to use BlackHole, your built-in
 *   speakers, and any other hardware output. 
 *   In the System Settings Sound panel, configure Input to BlackHole and Output to Multi-Channel Device.
 *   In Max's Audio Status control panel, set Input to BlackHole and Output to  Multi-Channel Device.
 *   
 * Run the AudioCapture application and start listening (spacebar turns listening on and off). 
 * Play a sound in Max. It should appear as pixel patterns in the AudioCapture window. 
 * 
 * If you are using the computer's mic and speakers, when the audio is streaming in and showing up
 * as animated pixels, you can click on it to hear the audio. Keep clicking and you can get feedback,
 * as the mic picks up the audio. For this reason, outboard audio is a good idea in most situations. 
 * 
 * In the Eclipse IDE in MacOS, inputs do not seem to function. Streaming from a file
 * works as expected, using Minim. Outputs should be set with the System Sound control panel. 
 *
 *
 * Press the 'p' key to play live audio from the built-in microphone or streaming from a file.
 * Press 't' to change the audio source from live audio to file or vice versa.
 * Press the spacebar to capture audio from the current stream into the audio buffer and write it to the screen.
 * Press 'k' to apply the hue and saturation in the colors array to mapImage.
 * Click on the image to play a sample.
 * 
 * Information about current source and streaming status is displayed in the application window. 
 *
 * See the MusicBoxBuffer sketch for a more advanced technique of moving through audio files, with more
 * efficient handling of audio and a complete user interface for drawing. You may find it useful to go
 * through the TutorialOne files first: MusicBoxBuffer is the culmination of the tutorial. 
 * 
 */

import java.util.Arrays;

import net.paulhertz.pixelaudio.*;
import net.paulhertz.pixelaudio.voices.*;
import processing.core.*;
import ddf.minim.*;
import ddf.minim.ugens.*;

/** PixelAudio library */
public PixelAudio pixelaudio;
/** A PixelMapGen to generate the signal path */
HilbertGen hGen;
/** A PixelAudioMapper to negotiate between image and audio */
PixelAudioMapper mapper;
/** the color channel we will draw to */
PixelAudioMapper.ChannelNames chan;

/** image for display */
PImage mapImage;
/** the number of pixels in the image, the number of samples in the audio signal, etc. */
int mapSize;
/** an array of colors through the RGB spectrum, handy for showing the shape of the signal path */
int[] colors;

/** Minim audio library */
Minim minim;
AudioInput audioIn;
AudioOutput audioOut;
/** A buffer for audio that we stream */
MultiChannelBuffer audioBuffer;
/** A Minim audio sampler to construct a sampling instrument */
Sampler audioSampler;
/** An audio recorder for capturing input from the microphone */
AudioRecorder recorder;
/** Our homemade audio sampling instrument */
PASamplerInstrument instrument;
/** A source for streaming audio from a file */
AudioPlayer anthem;
/** The class that captures audio for us */
StreamCapture streamCap;
/** audio signal as an array of floats */
float[] audioSignal;
/** audio signal transcoded to RGB data */
int[] rgbSignal;
int audioLength;
/** audio sampling rate */
int sampleRate = 44100;
/** duration of a sample played by the WFInstrument, in seconds */
float noteDuration = 0.5f;
/** actual length of a sample played by instrument */
int samplelen;
/** ADSR and parameters */
ADSRParams adsr;
float[] amplitudeRange = {0.4f, 0.9f};
float[] attackRange = {0.01f, 0.2f};
float[] decayRange = {0.05f, 0.3f};
float[] sustainRange = {0.4f, 0.8f};
float[] releaseRange = {0.1f, 0.4f};
/** audio variables */
boolean listening = false;
boolean listenLive = false;
/** interaction */
int pixelPos;
int samplePos;

boolean isVerbose = false;    // set true to get feedback in console


/**
 *  For the HilbertGen we require width == height == a power of 2. 
 */
public void settings() {
  size(512, 512);
}

public void setup() {
  pixelaudio = new PixelAudio(this);
  initMapper();
  mapSize = mapper.getSize();
  minim = new Minim(this);
  Sound.list();
  // set the devices you're using, depending on values from Sound.list()
  // works with Processing Sound library, but we're using Minim. 
  // In MacOS with Minim, use the System Settings control panel and AudioMIDI Setup.
  // You should be able to do something similar in Windows. 
  // Sound.inputDevice(6);
  // Sound.outputDevice(6);
  initAudio();
  showHelp();
}

public int[] getColors() {
  int[] colorWheel = new int[mapper.getSize()];
  pushStyle();
  colorMode(HSB, colorWheel.length, 100, 100);
  int h = 0;
  for (int i = 0; i < colorWheel.length; i++) {
    colorWheel[i] = color(h, 15, 80);
    h++;
  }
  popStyle();
  return colorWheel;
}

public void initMapper() {
  hGen = new HilbertGen(width, height);
  mapper = new PixelAudioMapper(hGen);
  chan = PixelAudioMapper.ChannelNames.L;
  mapImage = createImage(width, height, RGB);
  colors = getColors();
  mapImage.loadPixels();
  mapper.plantPixels(colors, mapImage.pixels, 0, mapper.getSize());
  mapImage.updatePixels();
}

public void draw() {
  image(mapImage, 0, 0);
  showAudioStatus();
  if (listening) {
    drawSignal();
  }
  drawInput();
}

public void mousePressed() {
  // get the position in the audio buffer that corresponds to the pixel location in the image
  pixelPos = mouseX + mouseY * width;
  samplePos = mapper.lookupSample(mouseX, mouseY);
  playSample(samplePos);
}

public void keyPressed() {
  switch (key) {
  case ' ':
    toggleListening();
    break;
  case 'p':
  case 'P':
    if (!listenLive) {
      if (anthem.isPlaying())
        anthem.pause();
      else
        anthem.loop();
    }
    break;
  case 'm':
    if (anthem.isMuted()) {
      anthem.unmute();
    }
    else {
      anthem.mute();
    }
    break;
  case 't':
    toggleAudioSource();
    break;
  case 'k': // apply the hue and saturation in the colors array to mapImage 
    mapImage.loadPixels();
    applyColor(colors, mapImage.pixels, mapper.getImageToSignalLUT());
    mapImage.updatePixels();
    break;
  case 'v': // turn verbose output to console on and off
    isVerbose = !isVerbose;
    if (isVerbose) println("---- isVerbose is now true, if it's false I won't say anything");
    break;
  case 'h':
    break;
  default:
    break;
  }
}

public void showHelp() {
  println(" * Press the 'p' key to play live audio from the built-in microphone or streaming from a file.");
  println(" * Press 't' to change the audio source from live audio to file or vice versa.");
  println(" * Press the spacebar to capture audio from the current stream into the audio buffer and write it to the screen.");
  println(" * Press 'k' to apply the hue and saturation in the colors array to mapImage.");
  println(" * Click on the image to play a sample.");
}

public void drawInput() {
  if (listenLive) {
    for (int i = 0; i < audioIn.bufferSize() - 1; i++) {
      line(i, 50 + audioIn.left.get(i) * 50, i + 1, 50 + audioIn.left.get(i + 1) * 50);
      line(i, 150 + audioIn.right.get(i) * 50, i + 1, 150 + audioIn.right.get(i + 1) * 50);
    }
  } else {
    for (int i = 0; i < anthem.bufferSize() - 1; i++) {
      line(i, 50 + anthem.left.get(i) * 50, i + 1, 50 + anthem.left.get(i + 1) * 50);
      line(i, 150 + anthem.right.get(i) * 50, i + 1, 150 + anthem.right.get(i + 1) * 50);
    }
  }
}

public void drawSignal() {
  rgbSignal = mapper.pluckSamplesAsRGB(audioSignal, 0, mapSize); // rgbSignal is now an array of rgb gray
  mapImage.loadPixels();
  mapper.plantPixels(rgbSignal, mapImage.pixels, 0, rgbSignal.length, chan);
  mapImage.updatePixels();
}  

/**
 * Utility method for applying hue and saturation values from a source array of RGB values
 * to the brightness values in a target array of RGB values, using a lookup table to redirect indexing.
 * 
 * @param colorSource    a source array of RGB data from which to obtain hue and saturation values
 * @param graySource     an target array of RGB data from which to obtain brightness values
 * @param lut            a lookup table, must be the same size as colorSource and graySource
 * @return the graySource array of RGB values, with hue and saturation values changed
 * @throws IllegalArgumentException if array arguments are null or if they are not the same length
 */
public int[] applyColor(int[] colorSource, int[] graySource, int[] lut) {
  if (colorSource == null || graySource == null || lut == null) 
    throw new IllegalArgumentException("colorSource, graySource and lut cannot be null.");
  if (colorSource.length != graySource.length || colorSource.length != lut.length) 
    throw new IllegalArgumentException("colorSource, graySource and lut must all have the same length.");
  // initialize a reusable array for HSB color data -- this is a way to speed up the applyColor() method
  float[] hsbPixel = new float[3];
  for (int i = 0; i < graySource.length; i++) {
    graySource[i] = PixelAudioMapper.applyColor(colorSource[lut[i]], graySource[i], hsbPixel);
  }
  return graySource;
}
