/*
 *  Copyright (c) 2024 - 2025 by Paul Hertz <ignotus@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as published
 *   by the Free Software Foundation; either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.paulhertz.pixelaudio;

import java.util.ArrayList;

/**
 * Generates a Moore curve over a square bitmap starting at (width/2 - 1, 0) and ending at (width/2, 0).
 * Width and height must be equal powers of 2. You can also call MooreGen(int depth) and width and height will equal Math.pow(2, depth).
 * The Moore curve is generated by rotating and reflecting 4 copies of a Hilbert curve with a depth of recursion one level lower.
 * 
 * @see PixelMapGen
 * @see HilbertGen
 * 
 */
public class MooreGen extends PixelMapGen {
	/** recursion depth */
	public int depth;

	public final static String description = "MooreGen generates a Moore curve over a square bitmap starting at (width/2 - 1, 0) and ending at (width/2, 0). "
			   + "\nWidth and height must be equal powers of 2. You can also call MooreGen(int depth) and width and height will equal Math.pow(2, depth). ";


	public MooreGen(int width, int height, AffineTransformType type) {
		super(width, height, type);
		this.depth = PixelMapGen.findPowerOfTwo(this.w);		// really handy to calculate depth before we generate the Moore curve
		// System.out.println("> MooreGen "+ width +", "+ height +", depth  = "+ depth + ", swap = "+ doXYSwap);
		this.generate();
	}

	public MooreGen(int width, int height) {
		this(width, height, AffineTransformType.NADA);
	}

	public MooreGen(int depth) {
		this( (int) Math.round(Math.pow(2, depth)), (int) Math.round(Math.pow(2, depth)), AffineTransformType.NADA);
	}

	public MooreGen(int depth, AffineTransformType type) {
		this( (int) Math.round(Math.pow(2, depth)), (int) Math.round(Math.pow(2, depth)), type );
	}


	@Override
	public String describe() {
		return MooreGen.description;
	}

	@Override
	public boolean validate(int width, int height) {
		if (width < 2) {
			System.out.println("MooreGen Error: 2 is the minimum value for width and height, 1 is the minimum value for depth.");
			return false;
		}
		if (width != height) {
			System.out.println("MooreGen Error: Width and height must be equal.");
			return false;
		}
		if (! PixelMapGen.isPowerOfTwo(width)) {
			System.out.println("MooreGen Error: Width and height must be equal to a power of 2.");
			return false;
		}
		return true;
	}

	@Override
	public int[] generate() {
		this.coords = this.generateCoordinates();
		// bitmap transforms of coordinates go here
		return this.setMapsFromCoords(this.coords);
	}
	
	
	/**
	 * @return		an ArrayList<int[]> of x, y coordinate pairs that are the points traversed by a generalized space-filling curve over a bitmap of dimensions w * h.
	 */
	private ArrayList<int[]> generateCoordinates() {
		return this.generateMooreCoordinates(this.getSize());
	}

	/**
	 * 
	 * @param n		the number of coordinate pairs to generate.
	 * @return		an ArrayList<int[]> of x, y coordinate pairs that are the points traversed by a generalized space-filling curve over a bitmap of dimensions w * h.
	 */
	private ArrayList<int[]> generateMooreCoordinates(int n) {
		ArrayList<int[]> mooreCoordinates = new ArrayList<>(n);
		int hilbDepth;
		if (n == 4) {
			mooreCoordinates.add(new int[] { 0, 0 });
			mooreCoordinates.add(new int[] { 0, 1 });
			mooreCoordinates.add(new int[] { 1, 1 });
			mooreCoordinates.add(new int[] { 1, 0 });
			System.out.println("-- Moore n == 4");
		} 
		else {
			hilbDepth = this.depth - 1;
			// transform four Hilbert curves to piece together a Moore curve
			HilbertGen hilb = new HilbertGen(hilbDepth);
			int[] xcoords = new int[hilb.size];
			int[] ycoords = new int[hilb.size];
			int i = 0;
			for (int[] xy : hilb.getCoordinates()) {
				xcoords[i] = xy[1];
				ycoords[i] = xy[0];
				i++;
			}
			int m = hilb.getWidth() - 1;
			// flip xcoords
			for (i = 0; i < xcoords.length; i++) {
				xcoords[i] = m - xcoords[i];
			}
			// now we fill mooreCoordinates with transformed copies of the Hilbert curve
			for (i = 0; i < hilb.size; i++) {
				mooreCoordinates.add(new int[] { xcoords[i], ycoords[i] });
			}
			for (i = 0; i < hilb.size; i++) {
				mooreCoordinates.add(new int[] { xcoords[i], ycoords[i] + hilb.w });
			}
			// flip xcoords
			for (i = 0; i < xcoords.length; i++) {
				xcoords[i] = m - xcoords[i];
			}
			// flip ycoords
			for (i = 0; i < ycoords.length; i++) {
				ycoords[i] = m - ycoords[i];
			}
			for (i = 0; i < hilb.size; i++) {
				mooreCoordinates.add(new int[] { xcoords[i] + hilb.w, ycoords[i] + hilb.w });
			}
			for (i = 0; i < hilb.size; i++) {
				mooreCoordinates.add(new int[] { xcoords[i] + hilb.w, ycoords[i] });
			}
		}
		return mooreCoordinates;
	}

}
